/*
 * Decompiled with CFR 0.152.
 */
package io.fabric.sdk.android.services.common;

import java.io.Closeable;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.RandomAccessFile;
import java.nio.channels.FileChannel;
import java.util.NoSuchElementException;
import java.util.logging.Level;
import java.util.logging.Logger;

public class QueueFile
implements Closeable {
    private static final Logger LOGGER = Logger.getLogger(QueueFile.class.getName());
    private static final int INITIAL_LENGTH = 4096;
    static final int HEADER_LENGTH = 16;
    private final RandomAccessFile raf;
    int fileLength;
    private int elementCount;
    private Element first;
    private Element last;
    private final byte[] buffer = new byte[16];

    public QueueFile(File file) throws IOException {
        if (!file.exists()) {
            QueueFile.initialize(file);
        }
        this.raf = QueueFile.open(file);
        this.readHeader();
    }

    QueueFile(RandomAccessFile raf) throws IOException {
        this.raf = raf;
        this.readHeader();
    }

    private static void writeInt(byte[] buffer, int offset, int value) {
        buffer[offset] = (byte)(value >> 24);
        buffer[offset + 1] = (byte)(value >> 16);
        buffer[offset + 2] = (byte)(value >> 8);
        buffer[offset + 3] = (byte)value;
    }

    private static void writeInts(byte[] buffer, int ... values) {
        int offset = 0;
        for (int value : values) {
            QueueFile.writeInt(buffer, offset, value);
            offset += 4;
        }
    }

    private static int readInt(byte[] buffer, int offset) {
        return ((buffer[offset] & 0xFF) << 24) + ((buffer[offset + 1] & 0xFF) << 16) + ((buffer[offset + 2] & 0xFF) << 8) + (buffer[offset + 3] & 0xFF);
    }

    private void readHeader() throws IOException {
        this.raf.seek(0L);
        this.raf.readFully(this.buffer);
        this.fileLength = QueueFile.readInt(this.buffer, 0);
        if ((long)this.fileLength > this.raf.length()) {
            throw new IOException("File is truncated. Expected length: " + this.fileLength + ", Actual length: " + this.raf.length());
        }
        this.elementCount = QueueFile.readInt(this.buffer, 4);
        int firstOffset = QueueFile.readInt(this.buffer, 8);
        int lastOffset = QueueFile.readInt(this.buffer, 12);
        this.first = this.readElement(firstOffset);
        this.last = this.readElement(lastOffset);
    }

    private void writeHeader(int fileLength, int elementCount, int firstPosition, int lastPosition) throws IOException {
        QueueFile.writeInts(this.buffer, fileLength, elementCount, firstPosition, lastPosition);
        this.raf.seek(0L);
        this.raf.write(this.buffer);
    }

    private Element readElement(int position) throws IOException {
        if (position == 0) {
            return Element.NULL;
        }
        this.raf.seek(position);
        return new Element(position, this.raf.readInt());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static void initialize(File file) throws IOException {
        File tempFile = new File(file.getPath() + ".tmp");
        try (RandomAccessFile raf = QueueFile.open(tempFile);){
            raf.setLength(4096L);
            raf.seek(0L);
            byte[] headerBuffer = new byte[16];
            QueueFile.writeInts(headerBuffer, 4096, 0, 0, 0);
            raf.write(headerBuffer);
        }
        if (!tempFile.renameTo(file)) {
            throw new IOException("Rename failed!");
        }
    }

    private static RandomAccessFile open(File file) throws FileNotFoundException {
        return new RandomAccessFile(file, "rwd");
    }

    private int wrapPosition(int position) {
        return position < this.fileLength ? position : 16 + position - this.fileLength;
    }

    private void ringWrite(int position, byte[] buffer, int offset, int count) throws IOException {
        if ((position = this.wrapPosition(position)) + count <= this.fileLength) {
            this.raf.seek(position);
            this.raf.write(buffer, offset, count);
        } else {
            int beforeEof = this.fileLength - position;
            this.raf.seek(position);
            this.raf.write(buffer, offset, beforeEof);
            this.raf.seek(16L);
            this.raf.write(buffer, offset + beforeEof, count - beforeEof);
        }
    }

    private void ringRead(int position, byte[] buffer, int offset, int count) throws IOException {
        if ((position = this.wrapPosition(position)) + count <= this.fileLength) {
            this.raf.seek(position);
            this.raf.readFully(buffer, offset, count);
        } else {
            int beforeEof = this.fileLength - position;
            this.raf.seek(position);
            this.raf.readFully(buffer, offset, beforeEof);
            this.raf.seek(16L);
            this.raf.readFully(buffer, offset + beforeEof, count - beforeEof);
        }
    }

    public void add(byte[] data) throws IOException {
        this.add(data, 0, data.length);
    }

    public synchronized void add(byte[] data, int offset, int count) throws IOException {
        QueueFile.nonNull(data, "buffer");
        if ((offset | count) < 0 || count > data.length - offset) {
            throw new IndexOutOfBoundsException();
        }
        this.expandIfNecessary(count);
        boolean wasEmpty = this.isEmpty();
        int position = wasEmpty ? 16 : this.wrapPosition(this.last.position + 4 + this.last.length);
        Element newLast = new Element(position, count);
        QueueFile.writeInt(this.buffer, 0, count);
        this.ringWrite(newLast.position, this.buffer, 0, 4);
        this.ringWrite(newLast.position + 4, data, offset, count);
        int firstPosition = wasEmpty ? newLast.position : this.first.position;
        this.writeHeader(this.fileLength, this.elementCount + 1, firstPosition, newLast.position);
        this.last = newLast;
        ++this.elementCount;
        if (wasEmpty) {
            this.first = this.last;
        }
    }

    public int usedBytes() {
        if (this.elementCount == 0) {
            return 16;
        }
        if (this.last.position >= this.first.position) {
            return this.last.position - this.first.position + 4 + this.last.length + 16;
        }
        return this.last.position + 4 + this.last.length + this.fileLength - this.first.position;
    }

    private int remainingBytes() {
        return this.fileLength - this.usedBytes();
    }

    public synchronized boolean isEmpty() {
        return this.elementCount == 0;
    }

    private void expandIfNecessary(int dataLength) throws IOException {
        int newLength;
        int elementLength = 4 + dataLength;
        int remainingBytes = this.remainingBytes();
        if (remainingBytes >= elementLength) {
            return;
        }
        int previousLength = this.fileLength;
        while ((remainingBytes += (previousLength = (newLength = previousLength << 1))) < elementLength) {
        }
        this.setLength(newLength);
        int endOfLastElement = this.wrapPosition(this.last.position + 4 + this.last.length);
        if (endOfLastElement < this.first.position) {
            FileChannel channel = this.raf.getChannel();
            channel.position(this.fileLength);
            int count = endOfLastElement - 4;
            if (channel.transferTo(16L, count, channel) != (long)count) {
                throw new AssertionError((Object)"Copied insufficient number of bytes!");
            }
        }
        if (this.last.position < this.first.position) {
            int newLastPosition = this.fileLength + this.last.position - 16;
            this.writeHeader(newLength, this.elementCount, this.first.position, newLastPosition);
            this.last = new Element(newLastPosition, this.last.length);
        } else {
            this.writeHeader(newLength, this.elementCount, this.first.position, this.last.position);
        }
        this.fileLength = newLength;
    }

    private void setLength(int newLength) throws IOException {
        this.raf.setLength(newLength);
        this.raf.getChannel().force(true);
    }

    public synchronized byte[] peek() throws IOException {
        if (this.isEmpty()) {
            return null;
        }
        int length = this.first.length;
        byte[] data = new byte[length];
        this.ringRead(this.first.position + 4, data, 0, length);
        return data;
    }

    public synchronized void peek(ElementReader reader) throws IOException {
        if (this.elementCount > 0) {
            reader.read(new ElementInputStream(this.first), this.first.length);
        }
    }

    public synchronized void forEach(ElementReader reader) throws IOException {
        int position = this.first.position;
        for (int i = 0; i < this.elementCount; ++i) {
            Element current = this.readElement(position);
            reader.read(new ElementInputStream(current), current.length);
            position = this.wrapPosition(current.position + 4 + current.length);
        }
    }

    private static <T> T nonNull(T t, String name) {
        if (t == null) {
            throw new NullPointerException(name);
        }
        return t;
    }

    public synchronized int size() {
        return this.elementCount;
    }

    public synchronized void remove() throws IOException {
        if (this.isEmpty()) {
            throw new NoSuchElementException();
        }
        if (this.elementCount == 1) {
            this.clear();
        } else {
            int newFirstPosition = this.wrapPosition(this.first.position + 4 + this.first.length);
            this.ringRead(newFirstPosition, this.buffer, 0, 4);
            int length = QueueFile.readInt(this.buffer, 0);
            this.writeHeader(this.fileLength, this.elementCount - 1, newFirstPosition, this.last.position);
            --this.elementCount;
            this.first = new Element(newFirstPosition, length);
        }
    }

    public synchronized void clear() throws IOException {
        this.writeHeader(4096, 0, 0, 0);
        this.elementCount = 0;
        this.first = Element.NULL;
        this.last = Element.NULL;
        if (this.fileLength > 4096) {
            this.setLength(4096);
        }
        this.fileLength = 4096;
    }

    @Override
    public synchronized void close() throws IOException {
        this.raf.close();
    }

    public boolean hasSpaceFor(int dataSizeBytes, int maxSizeBytes) {
        return this.usedBytes() + 4 + dataSizeBytes <= maxSizeBytes;
    }

    public String toString() {
        final StringBuilder builder = new StringBuilder();
        builder.append(this.getClass().getSimpleName()).append('[');
        builder.append("fileLength=").append(this.fileLength);
        builder.append(", size=").append(this.elementCount);
        builder.append(", first=").append(this.first);
        builder.append(", last=").append(this.last);
        builder.append(", element lengths=[");
        try {
            this.forEach(new ElementReader(){
                boolean first = true;

                @Override
                public void read(InputStream in, int length) throws IOException {
                    if (this.first) {
                        this.first = false;
                    } else {
                        builder.append(", ");
                    }
                    builder.append(length);
                }
            });
        }
        catch (IOException e) {
            LOGGER.log(Level.WARNING, "read error", e);
        }
        builder.append("]]");
        return builder.toString();
    }

    public static interface ElementReader {
        public void read(InputStream var1, int var2) throws IOException;
    }

    static class Element {
        static final int HEADER_LENGTH = 4;
        static final Element NULL = new Element(0, 0);
        final int position;
        final int length;

        Element(int position, int length) {
            this.position = position;
            this.length = length;
        }

        public String toString() {
            return this.getClass().getSimpleName() + "[" + "position = " + this.position + ", length = " + this.length + "]";
        }
    }

    private final class ElementInputStream
    extends InputStream {
        private int position;
        private int remaining;

        private ElementInputStream(Element element) {
            this.position = QueueFile.this.wrapPosition(element.position + 4);
            this.remaining = element.length;
        }

        @Override
        public int read(byte[] buffer, int offset, int length) throws IOException {
            QueueFile.nonNull(buffer, "buffer");
            if ((offset | length) < 0 || length > buffer.length - offset) {
                throw new ArrayIndexOutOfBoundsException();
            }
            if (this.remaining > 0) {
                if (length > this.remaining) {
                    length = this.remaining;
                }
                QueueFile.this.ringRead(this.position, buffer, offset, length);
                this.position = QueueFile.this.wrapPosition(this.position + length);
                this.remaining -= length;
                return length;
            }
            return -1;
        }

        @Override
        public int read() throws IOException {
            if (this.remaining == 0) {
                return -1;
            }
            QueueFile.this.raf.seek(this.position);
            int b = QueueFile.this.raf.read();
            this.position = QueueFile.this.wrapPosition(this.position + 1);
            --this.remaining;
            return b;
        }
    }
}

